<?php
/**
 * Plugin Name: Shortnr.pro — URL Shortener
 * Plugin URI: https://shortnr.pro
 * Description: Shorten URLs using your Shortnr.pro account directly from WordPress. Works with custom domains, QR codes, and analytics.
 * Version: 1.0.0
 * Author: Shortnr.pro
 * Author URI: https://shortnr.pro
 * License: GPL2
 * Text Domain: shortnr-pro
 */

if (!defined('ABSPATH')) exit;

define('SHORTNR_VERSION', '1.0.0');
define('SHORTNR_API_BASE', 'https://shortnr.pro/api');

// ─── Admin Settings Page ───────────────────────────────────────────────────

add_action('admin_menu', function () {
    add_options_page('Shortnr.pro Settings', 'Shortnr.pro', 'manage_options', 'shortnr-pro', 'shortnr_settings_page');
});

add_action('admin_init', function () {
    register_setting('shortnr_settings', 'shortnr_api_key', ['sanitize_callback' => 'sanitize_text_field']);
    register_setting('shortnr_settings', 'shortnr_domain', ['sanitize_callback' => 'sanitize_text_field']);
});

function shortnr_settings_page() {
    $api_key = get_option('shortnr_api_key', '');
    $domain  = get_option('shortnr_domain', '');
    ?>
    <div class="wrap">
        <h1 style="display:flex;align-items:center;gap:10px;">
            <img src="https://shortnr.pro/favicon.ico" width="24" height="24" />
            Shortnr.pro Settings
        </h1>
        <p>Enter your <a href="https://shortnr.pro/user/api" target="_blank">Shortnr.pro API key</a> to start shortening links from WordPress.</p>
        <form method="post" action="options.php">
            <?php settings_fields('shortnr_settings'); ?>
            <table class="form-table">
                <tr>
                    <th scope="row"><label for="shortnr_api_key">API Key</label></th>
                    <td>
                        <input type="text" id="shortnr_api_key" name="shortnr_api_key"
                            value="<?php echo esc_attr($api_key); ?>"
                            class="regular-text" placeholder="Your Shortnr.pro API key" />
                        <p class="description">Get your API key from <a href="https://shortnr.pro/user/api" target="_blank">shortnr.pro/user/api</a></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><label for="shortnr_domain">Custom Domain (optional)</label></th>
                    <td>
                        <input type="text" id="shortnr_domain" name="shortnr_domain"
                            value="<?php echo esc_attr($domain); ?>"
                            class="regular-text" placeholder="e.g. yourdomain.com" />
                        <p class="description">Leave blank to use shortnr.pro as the shortening domain.</p>
                    </td>
                </tr>
            </table>
            <?php submit_button('Save Settings'); ?>
        </form>

        <?php if ($api_key): ?>
        <hr>
        <h2>Quick Shorten</h2>
        <div style="max-width:500px;">
            <input type="url" id="shortnr-url-input" placeholder="Paste a long URL here..." class="regular-text" style="width:100%;margin-bottom:10px;" />
            <button id="shortnr-shorten-btn" class="button button-primary">Shorten URL</button>
            <div id="shortnr-result" style="margin-top:12px;display:none;">
                <strong>Short URL:</strong>
                <input type="text" id="shortnr-short-url" class="regular-text" readonly style="width:100%;margin-top:6px;" />
                <button class="button" onclick="document.getElementById('shortnr-short-url').select();document.execCommand('copy');" style="margin-top:6px;">Copy</button>
            </div>
            <div id="shortnr-error" style="margin-top:10px;color:red;display:none;"></div>
        </div>
        <script>
        document.getElementById('shortnr-shorten-btn').addEventListener('click', function() {
            var url = document.getElementById('shortnr-url-input').value.trim();
            if (!url) return;
            this.textContent = 'Shortening...';
            fetch('<?php echo admin_url('admin-ajax.php'); ?>', {
                method: 'POST',
                headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                body: 'action=shortnr_shorten&url=' + encodeURIComponent(url) + '&nonce=<?php echo wp_create_nonce('shortnr_nonce'); ?>'
            })
            .then(r => r.json())
            .then(data => {
                document.getElementById('shortnr-shorten-btn').textContent = 'Shorten URL';
                if (data.success) {
                    document.getElementById('shortnr-result').style.display = 'block';
                    document.getElementById('shortnr-error').style.display = 'none';
                    document.getElementById('shortnr-short-url').value = data.short_url;
                } else {
                    document.getElementById('shortnr-error').style.display = 'block';
                    document.getElementById('shortnr-error').textContent = data.message || 'Error shortening URL.';
                }
            });
        });
        </script>
        <?php endif; ?>
    </div>
    <?php
}

// ─── AJAX Handler ─────────────────────────────────────────────────────────

add_action('wp_ajax_shortnr_shorten', 'shortnr_ajax_shorten');
function shortnr_ajax_shorten() {
    check_ajax_referer('shortnr_nonce', 'nonce');
    if (!current_user_can('edit_posts')) wp_die('Unauthorized');

    $url     = esc_url_raw($_POST['url'] ?? '');
    $api_key = get_option('shortnr_api_key', '');
    $domain  = get_option('shortnr_domain', '');

    if (!$url || !$api_key) {
        wp_send_json(['success' => false, 'message' => 'Missing URL or API key.']);
    }

    $body = ['url' => $url, 'api' => $api_key];
    if ($domain) $body['domain'] = $domain;

    $response = wp_remote_post(SHORTNR_API_BASE . '/url/add', [
        'body'    => $body,
        'timeout' => 15,
    ]);

    if (is_wp_error($response)) {
        wp_send_json(['success' => false, 'message' => $response->get_error_message()]);
    }

    $data = json_decode(wp_remote_retrieve_body($response), true);

    if (!empty($data['shorturl'])) {
        wp_send_json(['success' => true, 'short_url' => $data['shorturl']]);
    } else {
        $msg = $data['message'] ?? 'Unknown error from Shortnr.pro API.';
        wp_send_json(['success' => false, 'message' => $msg]);
    }
}

// ─── Shortcode [shortnr url="..."] ────────────────────────────────────────

add_shortcode('shortnr', function ($atts) {
    $atts = shortcode_atts(['url' => ''], $atts, 'shortnr');
    if (!$atts['url']) return '';

    $api_key = get_option('shortnr_api_key', '');
    $domain  = get_option('shortnr_domain', '');
    if (!$api_key) return esc_url($atts['url']);

    $cache_key = 'shortnr_' . md5($atts['url']);
    $cached    = get_transient($cache_key);
    if ($cached) return esc_url($cached);

    $body = ['url' => $atts['url'], 'api' => $api_key];
    if ($domain) $body['domain'] = $domain;

    $response = wp_remote_post(SHORTNR_API_BASE . '/url/add', ['body' => $body, 'timeout' => 10]);
    if (is_wp_error($response)) return esc_url($atts['url']);

    $data = json_decode(wp_remote_retrieve_body($response), true);
    if (!empty($data['shorturl'])) {
        set_transient($cache_key, $data['shorturl'], WEEK_IN_SECONDS);
        return esc_url($data['shorturl']);
    }
    return esc_url($atts['url']);
});

// ─── Dashboard Widget ──────────────────────────────────────────────────────

add_action('wp_dashboard_setup', function () {
    wp_add_dashboard_widget('shortnr_widget', '🔗 Shortnr.pro — Quick Shorten', 'shortnr_dashboard_widget');
});

function shortnr_dashboard_widget() {
    $api_key = get_option('shortnr_api_key', '');
    if (!$api_key): ?>
        <p>⚠️ Add your API key in <a href="<?php echo admin_url('options-general.php?page=shortnr-pro'); ?>">Settings → Shortnr.pro</a></p>
    <?php else: ?>
        <input type="url" id="sw-url" placeholder="Paste a long URL..." style="width:100%;margin-bottom:8px;" class="widefat" />
        <button id="sw-btn" class="button button-primary" style="width:100%;">Shorten URL</button>
        <div id="sw-result" style="margin-top:8px;display:none;">
            <input type="text" id="sw-short" class="widefat" readonly />
            <button class="button" onclick="document.getElementById('sw-short').select();document.execCommand('copy');" style="margin-top:4px;width:100%;">Copy to Clipboard</button>
        </div>
        <p id="sw-error" style="color:red;display:none;margin-top:6px;"></p>
        <script>
        document.getElementById('sw-btn').addEventListener('click', function(){
            var url = document.getElementById('sw-url').value.trim();
            if(!url) return;
            this.textContent='Shortening...';
            fetch('<?php echo admin_url('admin-ajax.php'); ?>', {
                method:'POST',
                headers:{'Content-Type':'application/x-www-form-urlencoded'},
                body:'action=shortnr_shorten&url='+encodeURIComponent(url)+'&nonce=<?php echo wp_create_nonce('shortnr_nonce'); ?>'
            }).then(r=>r.json()).then(d=>{
                document.getElementById('sw-btn').textContent='Shorten URL';
                if(d.success){
                    document.getElementById('sw-result').style.display='block';
                    document.getElementById('sw-short').value=d.short_url;
                    document.getElementById('sw-error').style.display='none';
                } else {
                    document.getElementById('sw-error').style.display='block';
                    document.getElementById('sw-error').textContent=d.message;
                }
            });
        });
        </script>
    <?php endif;
}

// ─── Post Editor Meta Box ──────────────────────────────────────────────────

add_action('add_meta_boxes', function () {
    add_meta_box('shortnr_metabox', '🔗 Shortnr.pro', 'shortnr_metabox_html', ['post', 'page'], 'side', 'default');
});

function shortnr_metabox_html($post) {
    $api_key   = get_option('shortnr_api_key', '');
    $permalink = get_permalink($post->ID);
    $saved     = get_post_meta($post->ID, '_shortnr_short_url', true);
    ?>
    <?php if (!$api_key): ?>
        <p><a href="<?php echo admin_url('options-general.php?page=shortnr-pro'); ?>">Configure API key →</a></p>
    <?php else: ?>
        <?php if ($saved): ?>
            <p><strong>Short URL:</strong><br>
            <input type="text" value="<?php echo esc_url($saved); ?>" class="widefat" readonly onclick="this.select()" /></p>
        <?php endif; ?>
        <button type="button" id="mb-btn" class="button" style="width:100%;margin-top:4px;">
            <?php echo $saved ? 'Re-shorten This Post' : 'Shorten This Post URL'; ?>
        </button>
        <p id="mb-error" style="color:red;display:none;font-size:12px;"></p>
        <script>
        document.getElementById('mb-btn').addEventListener('click', function(){
            this.textContent = 'Shortening...';
            fetch('<?php echo admin_url('admin-ajax.php'); ?>', {
                method:'POST',
                headers:{'Content-Type':'application/x-www-form-urlencoded'},
                body:'action=shortnr_shorten&url=<?php echo urlencode($permalink); ?>&nonce=<?php echo wp_create_nonce('shortnr_nonce'); ?>'
            }).then(r=>r.json()).then(d=>{
                this.textContent='Re-shorten This Post';
                if(d.success){
                    document.getElementById('mb-error').style.display='none';
                    // save to hidden input so user can see it
                    var inp=document.createElement('input');
                    inp.type='text'; inp.className='widefat'; inp.readOnly=true;
                    inp.value=d.short_url; inp.onclick=function(){this.select()};
                    document.getElementById('mb-btn').before(inp);
                } else {
                    document.getElementById('mb-error').style.display='block';
                    document.getElementById('mb-error').textContent=d.message;
                }
            });
        });
        </script>
    <?php endif; ?>
    <?php
}
